#
# marc.pm : A Perl module to read and write MARC records
#
# Author: J.P.Knight@lut.ac.uk
#
# Description:
#	The MARC record is the external representation for works held in
#	library systems worldwide.  The intention of this module is to allow
#	such records to be read and written by Perl programs.  Currently it
#	only understands BLCMP MARC records but hopefully that will soon
#	change.
#
# $Id: Marc.pm,v 1.3 1997/01/12 17:55:21 jon Exp jon $
#

$VERSION = "0.01";

#
# MARC RECORD STRUCTURE:
#
# Acording to the BLCMP MARC User's Manual Rev.4.0 (Feb 1994) the structure
# of a MARC record is as follows:
#
# +--------+-----------+-------------------------+----------------------+
# | Leader | Directory | Variable Control Fields | Variable Data Fields |
# +--------+-----------+-------------------------+----------------------+
#
# The structure of the Leader is:
#
# Char. Pos.	Name
# 0-4		Record length
# 5		Record status
# 6		Type of record
# 7		Class of record
# 8-9		Blanks
# 10		Indicator count
# 11		Subfield mark count
# 12-16		Base address of data
# 17		Encoding level
# 18		Analytical record indicator
# 19		Source of record code
# 20		On UNION flag
# 21		SCP length from UKMARC
# 22		Blank
# 23		General record designator
#
#

package Metadata::Marc;
require Exporter;
@ISA = qw(Exporter);
#@EXPORT = qw(ReadMarcRecord WriteMarcRecord $MarcRecord );
@EXPORT = qw(ReadMarcRecord WriteMarcRecord );
@EXPORT_OK = qw();

# Developers debug switch
$debug = 0;

#
# Subroutine to read in a MARC record.
#
sub ReadMarcRecord {
  local($infile) = @_;
  local($RecordLength,$LeaderClass,$BaseAddress,$old_tag,$count,
    $Leader,$DirectoryEntries,%FIELD,$tag);

  # Reinitialise ourselves for the new record.
  $NewRecordStart = $NewRecordStart + $LastRecordLength;
  undef($MarcRecord);


  # Read in the leader
  if(read($infile,$Leader,24,0) != 24) {
    $MarcRecord = {
      marc_type => "Invalid",
    };
    warn "End of file!\n";
    return;
  }
  warn "Leader = $Leader\n" if $debug;
  $LastRecordLength = $RecordLength = substr($Leader,0,5);
  $LeaderClass = substr($Leader,7,1);
  $BaseAddress = substr($Leader,14,3);

  # Read in the directory entries
  $DirectoryEntries = &ReadDirectory($infile);

  # If we read in a valid directory, then read in all the fields.
  if($DirectoryEntries != 0) {
    undef(%FIELD);
    $old_tag = "";
    foreach $tag (sort (keys %DIR_LEVEL)) {
      $count = 0 if($old_tag != $tag);
      $old_tag = $tag;
      $FIELD{$tag}[$count] = &GrabField($tag,$count,$infile);
      warn "Value of field '$tag' is '$FIELD{$tag}[$count]'\n" if($debug);
      $count++;
    }

    $MarcRecord = {
      marc_type => "BLCMP",
      status => substr($Leader,5,1),
      type => substr($Leader,6,1),
      class => substr($Leader,7,1),
      indicator_count => substr($Leader,10,1),
      subfield_mark_count => substr($Leader,11,1),
      encoding_level => substr($Leader,17,1),
      analytical_record_ind => substr($Leader,18,1),
      source_of_record => substr($Leader,19,1),
      on_union_flag => substr($Leader,20,1),
      scp_length => substr($Leader,21,1),
      general_record_des => substr($Leader,23,1),
      data => { %FIELD },
      level => { %DIR_LEVEL },
    };
  } else {
    $MarcRecord = {
      marc_type => "BadDirectory",
      status => substr($Leader,5,1),
      type => substr($Leader,6,1),
      class => substr($Leader,7,1),
      indicator_count => substr($Leader,10,1),
      subfield_mark_count => substr($Leader,11,1),
      encoding_level => substr($Leader,17,1),
      analytical_record_ind => substr($Leader,18,1),
      source_of_record => substr($Leader,19,1),
      on_union_flag => substr($Leader,20,1), 
      scp_length => substr($Leader,21,1),    
      general_record_des => substr($Leader,23,1),
      data => { %FIELD },
      level => { %DIR_LEVEL },
    };
  }

  return($MarcRecord);
}

#
# Subroutine to read in the 001 field
#
sub GrabField {
    local($tag,$count,$infile) = @_;
    local($buffer,$control);

    seek($infile,$NewRecordStart+$BaseAddress+$DIR_START{$tag}[$count],0);
    read($infile,$buffer,$DIR_LENGTH{$tag}[$count]);
    $buffer =~ s/[\x1e\x1f\x1d]$//;
    return($buffer);
}

#
# Subroutine to read in the MARC directory
#
sub ReadDirectory {
  local($infile,$MarcRecord) = @_;
  local($FilePosition,$byte,$buffer,$tag,$entries);

  # Start a new directory
  undef(%DIR_LEVEL);
  undef(%DIR_LENGTH);
  undef(%DIR_START);
  undef(%TAG_COUNT);
  $TotalDirLength = 0;
  $entries = 0;

  # Get the current position of the file pointer
  $FilePosition = tell($infile);
  seek($infile,$FilePosition,0);
  warn "Starting directory read at file position 0".
    sprintf("%o", $FilePosition)."\n" if($debug);

  # Read the first character of the directory
  read($infile,$buffer,1);
  $byte = sprintf("%x",ord($buffer));

  # While the current character isn't the end of directory marker, read
  # in a directory entry.
  while($byte ne "1e") {
    warn "Reading directory entries from file position 0".
      sprintf("%o", $FilePosition)."\n" if($debug);
    warn "Byte = '$byte'\n" if($debug);

    seek($infile,$FilePosition,0);
    read($infile,$buffer,12);
    $tag = substr($buffer,0,3);
    $count=$TAG_COUNT{$tag};
    $TAG_COUNT{$tag}++;
    $DIR_LEVEL{$tag}[$count] = substr($buffer,3,1);
    $DIR_LENGTH{$tag}[$count] = substr($buffer,4,3);
    $DIR_START{$tag}[$count] = substr($buffer,7,5);

    warn "Buffer for '$tag' = '$buffer'\n" if($debug);
    warn "Directory Level for '$tag' = '$DIR_LEVEL{$tag}[$count]'\n" if($debug);
    warn "Directory Length for '$tag' = '$DIR_LENGTH{$tag}[$count]'\n" if($debug);
    warn "Directory Start for '$tag' = '$DIR_START{$tag}[$count]'\n" if($debug);

    # Keep a note of the total directory length read so far.
    $TotalDirLength += $DIR_LENGTH{$tag}[$count]+12;

    # Update the current character under consideration for being the 
    # end of directory marker.
    $FilePosition += 12;
    seek($infile,$FilePosition,0);
    read($infile,$buffer,1);
    $byte = sprintf("%x",ord($buffer));
    $entries++;
  }
  return($entries);
}

#
# Subroutine to write out a MARC record.
#
sub WriteMarcRecord {
  local($outfile) = @_;
  local($Directory,$DataFields,$Offset,$Length,$tag,@ThisTag,%DATA);

  $Directory = "";
  $DataFields = "";
  $Offset = 0;
  %DATA = %{ $MarcRecord->{data}};
  foreach $tag (sort(keys %DATA)) {
    @ThisTag = $MarcRrecord->{data}{$tag};
    $number = @ThisTag;
    $count = 0;
    while($count != $number) {
      $Length=length($MarcRecord->{data}{$tag}[$count]);
      $level = $MarcRecord->{level}{$tag}[$count];
      $Entry=sprintf("%0.3d%0.1d%0.3d%0.5d",$tag,$level,$Length+1,$Offset);
      $Offset = $Offset + $Length + 1; 
      $Directory = $Directory . $Entry;
      $DataFields = $DataFields . $MarcRecord->{data}{$tag}[$count] ."\x1e";
      $count++;
    }
  }
  $DataFields =~ s/\x1e$//;
  $DataFields = $DataFields . "\x1d";
  $Directory = $Directory."\x1e";

  $RecordLength = 24+length($Directory)+length($DataFields);
  $BaseAddress = 24+length($Directory);
  $Leader = sprintf("%0.5d%0.1d%1s%1s  %s%s%0.5d%1s%1s%1s%1s%1s %1s",
     $RecordLength,$MarcRecord->{status},$MarcRecord->{type},
     $MarcRecord->{class},$MarcRecord->{indicator_count},
     $MarcRecord->{subfield_mark_count},$BaseAddress,
     $MarcRecord->{encoding_level},$MarcRecord->{analytical_record_ind},
     $MarcRecord->{source_of_record},$MarcRecord->{on_union_flag},
     $MarcRecord->{scp_length},$MarcRecord->{general_record_des});

  print $outfile "$Leader$Directory$DataFields";

}
1;
